import os
import pickle
import numpy as np
import pandas as pd  # Ensure pandas is installed
from flask import Blueprint, request, jsonify

# ✅ Define the blueprint
detect_bp = Blueprint("detect", __name__)

# ✅ Path to the trained model
MODEL_PATH = os.path.abspath(os.path.join("models", "anomaly_detector.pkl"))

# ✅ Load the trained model
if not os.path.exists(MODEL_PATH):
    raise FileNotFoundError(f"❌ Model not found at: {MODEL_PATH}")

with open(MODEL_PATH, "rb") as f:
    model = pickle.load(f)

print("✅ Intrusion Detection Model Loaded Successfully!")

# ✅ Define feature names (Ensure they match your training dataset)
FEATURE_NAMES = [
    "url_length",
    "has_https",
    "num_dots",
    "is_shortened",
    "contains_suspicious_words",
    "mouse_movement_anomaly",
    "keystroke_pattern"
]

@detect_bp.route("/api/detect", methods=["POST"])
def detect_intrusion():
    try:
        # ✅ Get request data
        data = request.get_json()

        if not data:
            return jsonify({"error": "No input data provided"}), 400
        
        # ✅ Ensure all required features are present
        missing_features = [feature for feature in FEATURE_NAMES if feature not in data]
        if missing_features:
            return jsonify({"error": f"Missing features: {missing_features}"}), 400

        # ✅ Convert JSON to DataFrame (to match model training format)
        features_df = pd.DataFrame([data], columns=FEATURE_NAMES)

        # ✅ Make prediction
        prediction = model.predict(features_df)[0]

        # ✅ Interpret result
        result = "Malicious" if prediction == 1 else "Safe"

        return jsonify({"prediction": int(prediction), "status": result})

    except Exception as e:
        return jsonify({"error": str(e)}), 500
