document.addEventListener("DOMContentLoaded", function () {
    let checkButton = document.getElementById("checkIntrusion");

    if (checkButton) {
        checkButton.addEventListener("click", detectIntrusion);
    } else {
        console.error("❌ Error: Element #checkIntrusion not found!");
    }
});

async function detectIntrusion() {
    try {
        let tab = await getCurrentTab();
        if (!tab || !tab.url) {
            console.error("❌ No active tab or URL found!");
            alert("❌ Error: Unable to detect the current website.");
            return;
        }

        let url = tab.url;
        console.log(`🔍 Checking URL: ${url}`);

        let features = extractFeatures(url);
        console.log("📊 Extracted Features:", features);

        let response = await fetch("http://127.0.0.1:5000/api/detect", {
            method: "POST",
            headers: { "Content-Type": "application/json" },
            body: JSON.stringify(features),
        });

        if (!response.ok) {
            throw new Error(`Server responded with ${response.status}`);
        }

        let result = await response.json();
        console.log("📡 Intrusion Detection Result:", result);

        if (result.prediction === 1) {
            alert("🚨 Warning: This website might be malicious!");
        } else {
            alert("✅ This website appears safe.");
        }
    } catch (error) {
        console.error("❌ Error in detection:", error);
        alert("❌ Error connecting to the detection system. Check console for details.");
    }
}

function getCurrentTab() {
    return new Promise((resolve, reject) => {
        chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
            if (chrome.runtime.lastError) {
                console.error("❌ Chrome API Error:", chrome.runtime.lastError);
                reject("Chrome API Error");
            }
            if (!tabs || tabs.length === 0) {
                reject("❌ No active tab found.");
            }
            resolve(tabs[0]);
        });
    });
}

function extractFeatures(url) {
    return {
        "url_length": url.length,
        "has_https": url.startsWith("https") ? 1 : 0,
        "num_dots": (url.match(/\./g) || []).length,
        "is_shortened": /bit\.ly|goo\.gl|t\.co|tinyurl\.com/.test(url) ? 1 : 0,
        "contains_suspicious_words": /free|win|prize|bonus|claim/.test(url) ? 1 : 0,
        "mouse_movement_anomaly": 0, // Placeholder (Needs real-time tracking)
        "keystroke_pattern": 0 // Placeholder (Needs key event analysis)
    };
}
